/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file WE17Driver.cxx
    \author Romain BONDUE
    \date 05/07/2005
    \note Many functions here could have been "inline", but since they are also
          "virtual", it's almost useless. */
#include <sstream>
#include <cmath> // pow()
#include <cstring> // strncpy()
#include <cerrno>
#include <algorithm> // sort()
#ifndef NDEBUG
#include <iostream>
#endif // NDEBUG

extern "C"{
#include <net/if_arp.h>
}

#include "WE17Driver.h"
#include "ErrorsCode.h"



namespace
{
    inline nsWireless::CQuality iw_qualityToCQuality (const ::iw_quality& Iw)
                                                                        throw()
    {
            // There's an unusual 8 bits arithmetic, that's why the -256.
        return nsWireless::CQuality (Iw.qual, -256 + Iw.level,
                                     -256 + Iw.noise);

    } // iw_qualityToCQuality()


    inline nsWireless::CEncryptionD MakeCEncryption (const char* Key,
                                            unsigned Length, int Flags) throw()
    {
        return nsWireless::CEncryptionD (Flags & IW_ENCODE_DISABLED
                                        ? nsWireless::CHexaKey()
                                        : nsWireless::CHexaKey (Key, Length),
                            Flags & IW_ENCODE_RESTRICTED ? nsWireless::Shared
                                                         : nsWireless::Open,
                                Flags & IW_ENCODE_DISABLED ? nsWireless::None
                                                           : nsWireless::WEP);

    } // MakeCEncryption()


    std::string CatStringNInt (std::string Str, int Number) throw()
    {
        std::ostringstream Os;
        Os << Str << Number;
        return Os.str();

    } // CatStringNInt()


    inline double iw_freqToDouble (::iw_freq Freq) throw()
    {
        return Freq.m / pow (10, 9 - Freq.e);

    } // iw_freqToDouble()


    const unsigned TxUnit (1000);

} // anonymous namespace



nsWireless::CWE17Driver::CWE17Driver (const std::string& Name)
                                                throw (nsErrors::CSystemExc)
    : m_Data (m_Req.u)
{
        // Starting from here, we just collect informations :
    Name.copy (m_Req.ifr_ifrn.ifrn_name, IFNAMSIZ);
    m_Req.ifr_ifrn.ifrn_name [Name.size() > IFNAMSIZ - 1 ? IFNAMSIZ - 1
                                                         : Name.size()] = '\0';
    Ioctl (SIOCGIWNAME, "Can't get protocol/provider name.");
    m_ProtoName = m_Req.u.name;

    ::iw_range Range;
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (&Range);
    m_Req.u.data.length = sizeof (Range);
    m_Req.u.data.flags = 0;
    try
    {       // Only used to get channels values.
        Ioctl (SIOCGIWRANGE, "Can't get interface informations.");
        for (unsigned i (0) ; i < Range.num_frequency ; ++i)
            m_SupportedFreqVec.push_back (CFreq (Range.freq [i].i,
                                         iw_freqToDouble (Range.freq [i])));
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
#ifndef NDEBUG
        std::cerr << "CWE17Driver::CWE17Driver() Can't get interface"
                     " information : " << Exc.GetCode() << std::endl;
#endif // NDEBUG
        if (Exc.GetCode() != EOPNOTSUPP && Exc.GetCode() != ENETDOWN) throw;
    }
    if (m_SupportedFreqVec.empty())
    {
#ifndef NDEBUG
        std::cerr << "CWE17Driver::CWE17Driver() generate channel list."
                  << std::endl;
#endif // NDEBUG
        for (unsigned i (1) ; i < 14 ; ++i) // Let's put the whole spectrum.
            m_SupportedFreqVec.push_back (CFreq (i, 2.412 + (i - 1) * 0.005));
        m_SupportedFreqVec.push_back (CFreq (14, 2.484));
    }

} // CWE17Driver()


std::string nsWireless::CWE17Driver::GetSSID () const
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetSSID()" << std::endl;
#endif // NDEBUG
    char Buffer [IW_ESSID_MAX_SIZE];
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Req.u.data.length = IW_ESSID_MAX_SIZE;
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWESSID, "Can't get SSID.");
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetSSID() : "
              << std::string (reinterpret_cast<const char*>
                  (m_Req.u.data.pointer), 0, m_Req.u.data.length) << std::endl;
#endif // NDEBUG
    if (!m_Req.u.data.length) return std::string();
    return std::string (Buffer, 0, m_Req.u.data.length);

} // GetSSID()


    // No control on the SSID, however it'll be truncated if too long.
void nsWireless::CWE17Driver::SetSSID (const std::string& SSID)
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetSSID() " << SSID << std::endl;
#endif // NDEBUG
    m_Req.u.data.pointer = const_cast< ::caddr_t> (SSID.c_str());
#if WIRELESS_EXT < 21
    m_Req.u.data.length = SSID.size() + 1 > IW_ESSID_MAX_SIZE
                                                          ? IW_ESSID_MAX_SIZE  
                             /* + 1 for the '\0'. */      : SSID.size() + 1;
#else
    m_Req.u.data.length = SSID.size() > IW_ESSID_MAX_SIZE ? IW_ESSID_MAX_SIZE  
                                                          : SSID.size();
#endif // WIRELESS_EXT
    m_Req.u.data.flags = 1; // Only use *this* SSID.
    Ioctl (SIOCSIWESSID, std::string ("Can't set SSID : ") + SSID);

} // SetSSID()


unsigned nsWireless::CWE17Driver::GetChannel () const
                                                throw (nsErrors::CException)
{
    const double Value (GetDriverFreqValue());
    if (Value > 0) return GetMatchingFreq (0, Value).GetChannel();
    return unsigned (-Value);

} // GetChannel()


void nsWireless::CWE17Driver::SetChannel (unsigned Channel)
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetChannel() " << Channel << std::endl;
#endif // NDEBUG
    m_Req.u.freq.m = Channel;
    m_Req.u.freq.e = 0;
#if WIRELESS_EXT > 16
    m_Req.u.freq.flags = IW_FREQ_FIXED;
#endif
    Ioctl (SIOCSIWFREQ, CatStringNInt ("Can't set channel : ", Channel));

} // SetChannel()


double nsWireless::CWE17Driver::GetFrequency () const
                                                throw (nsErrors::CException)
{
    const double Value (GetDriverFreqValue());
    if (Value > 0) return Value;
    return GetMatchingFreq (unsigned (-Value)).GetFrequency();

} // GetFrequency()


void nsWireless::CWE17Driver::SetFrequency (double F)
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetFrequency() " << F << std::endl;
#endif // NDEBUG
    m_Req.u.freq.m = unsigned (F * 1e8);
    m_Req.u.freq.e = 1;
#if WIRELESS_EXT > 16
    m_Req.u.freq.flags = IW_FREQ_FIXED;
#endif
    Ioctl (SIOCSIWFREQ, CatStringNInt ("Can't set frequency : ",
                                       m_Req.u.freq.m));

} // SetFrequency()


nsWireless::CEncryptionD nsWireless::CWE17Driver::GetEncryption ()
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetEncryption()" << std::endl;
#endif // NDEBUG
        /* ? FIXME ? The key index is not considered and only the first key
                     is reported. */
    char Buffer [IW_ENCODING_TOKEN_MAX];
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (Buffer);
    m_Req.u.data.length = IW_ENCODING_TOKEN_MAX;
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWENCODE, "Can't get WEP status.");
    return MakeCEncryption (Buffer, m_Req.u.data.length, m_Req.u.data.flags);

} // GetEncryption()


void nsWireless::CWE17Driver::SetEncryption (const CEncryptionD& Descriptor)
                                throw (nsErrors::CSystemExc, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetEncryption() "
              << GetAuthName (Descriptor.GetAuth()) << '\t'
              << GetEncryptName (Descriptor.GetEncrypt());
    for (unsigned i (0) ; i < CEncryptionD::MaxNbKey ; ++i)
        std::cerr << "\n\t" << i << " : " << Descriptor.GetKey (i).GetStr();
    std::cerr << std::endl;
#endif // NDEBUG
    if (Descriptor.GetEncrypt() == nsWireless::WEP)
    {
        const int AuthFlag (Descriptor.GetAuth() ==
                                    nsWireless::Shared ? IW_ENCODE_RESTRICTED
                                                       : IW_ENCODE_OPEN);
        for (unsigned I (CEncryptionD::MaxNbKey) ; I ; )
        {
            m_Req.u.data.flags = AuthFlag;
            m_Req.u.data.flags |= I;
            if (Descriptor.GetKey (--I).Empty()) // Decremented here.
            {
                m_Req.u.data.pointer = 0;
                m_Req.u.data.length = 0;
                m_Req.u.data.flags |= IW_ENCODE_NOKEY | IW_ENCODE_DISABLED;
            }
            else
            {
                m_Req.u.data.pointer = const_cast< ::caddr_t>
                                                (Descriptor.GetKey (I).Get());
                m_Req.u.data.length = Descriptor.GetKey (I).Size();
            }
            Ioctl (SIOCSIWENCODE,
            CatStringNInt (std::string ("Can't set WEP encryption with key : ")
                               + Descriptor.GetKey (I).GetStr() + " Index : ",
                           I));
        }
        try   // Now activate the default key, it may fail because previous
        {     // calls may have already activated it with some driver.
            m_Req.u.data.pointer = 0;
            m_Req.u.data.length = 0;
            m_Req.u.data.flags = AuthFlag | IW_ENCODE_NOKEY |
                                            (Descriptor.GetDefaultKey() + 1);
            Ioctl (SIOCSIWENCODE, "Can't activate default key.");
        }
        catch (const nsErrors::CException& Exc)
        {
#ifndef NDEBUG
            std::cerr << Exc.GetMsg() << ' ' << Exc.GetCode() << std::endl;
#endif // NDEBUG
        }
    }
    else
    {
        m_Req.u.data.flags = IW_ENCODE_NOKEY | IW_ENCODE_DISABLED;
        m_Req.u.data.pointer = 0;
        m_Req.u.data.length = 0;
        Ioctl (SIOCSIWENCODE, "Can't disable WEP encryption.");
    }

} // SetEncryption()


nsWireless::Mode_e nsWireless::CWE17Driver::GetMode () const
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetMode()" << std::endl;
#endif // NDEBUG
    Ioctl (SIOCGIWMODE, "Can't get operating mode.");
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetMode() " << GetModeName (Mode_e
                                                (m_Req.u.mode)) << std::endl;
#endif // NDEBUG
    return Mode_e (m_Req.u.mode);

} // GetMode()


void nsWireless::CWE17Driver::SetMode (Mode_e Mode)
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetMode() " << GetModeName (Mode) << std::endl;
#endif // NDEBUG
    m_Req.u.mode = Mode;
    Ioctl (SIOCSIWMODE,
           std::string ("Can't set operating mode : ") + GetModeName (Mode));

} // SetMode()


nsWireless::CQuality nsWireless::CWE17Driver::GetQuality () const
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetQuality()" << std::endl;
#endif // NDEBUG
    ::iw_statistics Stat;
    m_Req.u.data.pointer = reinterpret_cast < ::caddr_t> (&Stat);
    m_Req.u.data.length = sizeof (Stat);
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWSTATS, "Can't get signal quality.");
    return iw_qualityToCQuality (Stat.qual);

} // GetQuality()


int nsWireless::CWE17Driver::GetTxRate () const throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetTxRate()" << std::endl;
#endif // NDEBUG
    Ioctl (SIOCGIWRATE, "Can't get TX rate.");
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetTxRate() " << m_Req.u.bitrate.value
              << std::endl;
#endif // NDEBUG
    return m_Req.u.bitrate.value / TxUnit;

} // GetTxRate()


void nsWireless::CWE17Driver::SetTxRate (int TxRate)
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetTxRate() " << TxRate << std::endl;
#endif // NDEBUG
    m_Req.u.bitrate.value = TxRate * TxUnit;
    m_Req.u.bitrate.fixed = 0;  // Hardware should use auto-select.
    m_Req.u.bitrate.disabled = 0;
    m_Req.u.bitrate.flags = 0;
    Ioctl (SIOCSIWRATE, CatStringNInt ("Can't set TX rate : ", TxRate));

} // SetTxRate()


nsWireless::CMacAddress nsWireless::CWE17Driver::GetAPMacAddr () const
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetAPMacAddr()" << std::endl;
#endif // NDEBUG
    try {Ioctl (SIOCGIWAP, "Can't get access point Mac address.");}
    catch (const nsErrors::CSystemExc& Exc)
    {     // I don't consider this an error.
#ifndef NDEBUG
        std::cerr << Exc.GetMsg() << ' ' << Exc.GetCode() << std::endl;
#endif // NDEBUG
        if (Exc.GetCode() == ENOTCONN) return CMacAddress();
        throw;
    }
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetAPMacAddr() " << std::hex;
    const unsigned char* const RawData (reinterpret_cast<const unsigned char*>
                                                    (m_Req.u.ap_addr.sa_data));
    for (unsigned i (0) ; i < 6 ; ++i)
        std::cerr << unsigned (RawData [i]) << ' ';
    std::cerr << std::dec << std::endl;
#endif // NDEBUG
    return CMacAddress (m_Req.u.ap_addr.sa_data);

} // GetAPMacAddr()


void nsWireless::CWE17Driver::SetAPMacAddr (const CMacAddress& Addr)
                                throw (std::bad_alloc, nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::SetAPMacAddr() " << Addr.GetStr() << std::endl;
#endif // NDEBUG
    m_Req.u.ap_addr.sa_family = ARPHRD_ETHER;
    memcpy (m_Req.u.ap_addr.sa_data, Addr.Get(), CMacAddress::Size);
    Ioctl (SIOCSIWAP, std::string ("Can't set access point Mac address : ") +
                      Addr.GetStr());

} // SetAPMacAddr()


void nsWireless::CWE17Driver::Scan () throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::Scan()" << std::endl;
#endif // NDEBUG
    m_Req.u.data.pointer = 0;
    m_Req.u.data.length = 0;
        // Full scanning.
    m_Req.u.data.flags = IW_SCAN_ALL_ESSID | IW_SCAN_ALL_FREQ |
                         IW_SCAN_ALL_MODE | IW_SCAN_ALL_RATE;
    Ioctl (SIOCSIWSCAN, "Can't trigger scanning.");

} // Scan()


void nsWireless::CWE17Driver::GetScanResult (std::vector<CCell>& CellVec) const
                                throw (nsErrors::CException, std::bad_alloc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetScanResult()" << std::endl;
#endif // NDEBUG
    m_Req.u.data.flags = 0;
    std::vector<char> Buffer (256); // In iwlist, the starting size is 128.
    for ( ; ; )
    {
        m_Req.u.data.pointer = reinterpret_cast < ::caddr_t> (&Buffer [0]);
        m_Req.u.data.length = Buffer.size();
        try
        {
            Ioctl (SIOCGIWSCAN, "Can't get scan results.");
            break;
        }
        catch (const nsErrors::CSystemExc& Exc)
        {
#ifndef NDEBUG
            std::cerr << "\n\tCWE17Driver::GetScanResult() failed" << std::endl;
#endif // NDEBUG
            if (Exc.GetCode() == E2BIG)
            {       // If the driver gives the expected size.
#ifndef NDEBUG
                std::cerr << " : Buffer too small" << std::endl;
#endif // NDEBUG
                if (m_Req.u.data.length > Buffer.size())
                    Buffer.resize (m_Req.u.data.length);
                    // Else double the size.
                else
                    Buffer.resize (Buffer.size() * 2U);
            }
            else
                throw;
        }
    }
        // Things are MUCH more complicated in iwlib.c...
    const char* const pEvents (&Buffer [0]);
    for (unsigned Offset (0) ; Offset < m_Req.u.data.length ; )
        Offset += FillCellVec (reinterpret_cast<const ::iw_event*> 
                                                (pEvents + Offset), CellVec);

} // GetScanResult()


unsigned nsWireless::CWE17Driver::FillCellVec (const ::iw_event* pEvent,
                                               std::vector<CCell>& Vec) const
                                                throw (nsErrors::CException)
{
#ifndef NDEBUG
    std::cerr << "\n\tEvent : " << std::hex << pEvent->cmd << std::dec;
#endif // NDEBUG
        // Jean Tourrilhes in iwlib.c warns that data may be unaligned.
    switch (pEvent->cmd)
    {
      case SIOCGIWAP :
      {
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWAP : " << std::hex;
        const unsigned char* const RawData
                                        (reinterpret_cast<const unsigned char*>
                                                    (m_Req.u.ap_addr.sa_data));
        for (unsigned i (0) ; i < 6 ; ++i)
            std::cerr << unsigned (RawData [i]) << ' ';
        std::cerr << std::dec << std::endl;
#endif // NDEBUG
        Vec.push_back (CCell (CMacAddress (pEvent->u.ap_addr.sa_data)));
      }
      break;

      case SIOCGIWMODE :
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWMODE : "
                  << GetModeName (Mode_e (pEvent->u.mode));
#endif // NDEBUG
        Vec.back().m_Mode = Mode_e (pEvent->u.mode);
      break;

      case SIOCGIWESSID :
      {
          /* From wireless.h (v19) : iw_point events are special.
           * First, the payload (extra data) come at the end of the event,
           * so they are bigger than IW_EV_POINT_LEN.
           * Second, we omit the pointer, so start at an offset. */
#if WIRELESS_EXT > 18
        const unsigned CstLength (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->length);
#else
        const unsigned CstLength (pEvent->u.essid.length);
#endif
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWESSID length : " << CstLength << ' ';
#endif // NDEBUG
        if (CstLength)
            /* From wireless.h (v17) : "Note : in the case of iw_point,
             * the extra data will come at the end of the event". */
            Vec.back().m_SSID = std::string (reinterpret_cast<const char*>
                                                    (pEvent) + IW_EV_POINT_LEN,
                                             0, CstLength);
#ifndef NDEBUG
        std::cerr << '#' << Vec.back().GetSSID() << '#';
#endif // NDEBUG
      }
      break;

      case SIOCGIWENCODE :
      {
          /* From wireless.h (v19) : iw_point events are special.
           * First, the payload (extra data) come at the end of the event,
           * so they are bigger than IW_EV_POINT_LEN.
           * Second, we omit the pointer, so start at an offset. */
#if WIRELESS_EXT > 18
        const unsigned CstLength (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->length);
        unsigned Flags (reinterpret_cast<const ::iw_point*>
        (reinterpret_cast<const char*> (&pEvent->u) - sizeof (void*))->flags);
#else
        const unsigned CstLength (pEvent->u.essid.length);
        unsigned Flags (pEvent->u.essid.flags);
#endif
        const char* Data (0);
        if (CstLength)
            Data = reinterpret_cast<const char*> (pEvent) + IW_EV_POINT_LEN;
        else Flags |= IW_ENCODE_NOKEY;
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWENCODE length : " << CstLength << " flags : "
                  << Flags;;
#endif // NDEBUG
        Vec.back().m_EncDescriptor = MakeCEncryption (Data, CstLength, Flags);
#ifndef NDEBUG
        std::cerr << "\n\t"
                  << GetAuthName (Vec.back().GetEncryptionD().GetAuth()) << ' '
                  << GetEncryptName (Vec.back().GetEncryptionD().GetEncrypt());
#endif // NDEBUG
      }
      break;

      case SIOCGIWFREQ :
      {
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWFREQ : " << pEvent->u.freq.m << ' '
                  << pEvent->u.freq.e << ' ' << pEvent->u.freq.i << ' '
                  << pEvent->u.freq.flags << std::endl;
#endif // NDEBUG
        if (pEvent->u.freq.m > 1000)
            Vec.back().m_Channel = GetMatchingFreq (0,
                                iw_freqToDouble (pEvent->u.freq)).GetChannel();
        else Vec.back().m_Channel = pEvent->u.freq.m;
      }
      break;

      case IWEVQUAL :
#ifndef NDEBUG
        std::cerr << "\n\tIWEVQUAL";
#endif // NDEBUG
        Vec.back().m_Quality = iw_qualityToCQuality (pEvent->u.qual);
      break;

      case SIOCGIWRATE :
#ifndef NDEBUG
        std::cerr << "\n\tSIOCGIWRATE " << pEvent->u.bitrate.value;
#endif // NDEBUG
        Vec.back().m_Rate =  pEvent->u.bitrate.value / TxUnit;
      break;

#ifndef NDEBUG
      default :
        std::cerr << " Unknown.";
      //break;
#endif // NDEBUG
    }
#ifndef NDEBUG
    std::cerr << "\n\tEvent length : " << pEvent->len << std::endl;
#endif // NDEBUG
    return pEvent->len;

} // FillCellVec()


nsWireless::CFreq nsWireless::CWE17Driver::GetMatchingFreq
                            (unsigned Channel, double Frequency) const throw()
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetMatchingFreq() " << Channel << ' '
              << Frequency << " : ";
#endif // NDEBUG
    for (unsigned i (m_SupportedFreqVec.size()) ; i-- ; )
        if (m_SupportedFreqVec [i].GetChannel() == Channel ||
                            m_SupportedFreqVec [i].GetFrequency() == Frequency)
        {
#ifndef NDEBUG
            std::cerr << "matched.\n";
#endif // NDEBUG
            return m_SupportedFreqVec [i];
        }
#ifndef NDEBUG
    std::cerr << "no match.\n";
#endif // NDEBUG
    return CFreq (0U, 0.0);

} // GetMatchingFreq()


double nsWireless::CWE17Driver::GetDriverFreqValue () const
                                                throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetDriverFreqValue()\n";
#endif // NDEBUG
    Ioctl (SIOCGIWFREQ, "Can't get frequency/channel.");
    if (m_Req.u.freq.m > 1000)
        return iw_freqToDouble (m_Req.u.freq);
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetDriverFreqValue " << m_Req.u.freq.m << ' '
              << m_Req.u.freq.e << ' ' << m_Req.u.freq.i << ' '
              << m_Req.u.freq.flags << std::endl;
#endif // NDEBUG
    return -m_Req.u.freq.m;

} // GetDriverFreqValue())


unsigned nsWireless::CWE17Driver::GetPrivateIoctls
        (::iw_priv_args* Tab, unsigned TabSize) throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetPrivateIoctls()\n";
#endif // NDEBUG
    m_Data.data.pointer = reinterpret_cast< ::caddr_t> (Tab);
    m_Data.data.length = TabSize;
    m_Data.data.flags = 0;
    Ioctl (SIOCGIWPRIV, "Can't get private ioctl interface.");

    return m_Data.data.length;

} // GetPrivateIoctl()


void nsWireless::CWE17Driver::Commit () throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::Commit()\n";
#endif // NDEBUG
    m_Data.data.pointer = 0;
    m_Data.data.length = 0;
    m_Data.data.flags = 0;
    Ioctl (SIOCSIWCOMMIT, "Can't commit.");

} // Commit()


void nsWireless::CWE17Driver::GetSupportedRates (std::vector<int>& RatesVec)
                                            const throw (nsErrors::CSystemExc)
{
#ifndef NDEBUG
    std::cerr << "CWE17Driver::GetSupportedRates()\n";
#endif // NDEBUG
    ::iw_range Range;
    m_Req.u.data.pointer = reinterpret_cast< ::caddr_t> (&Range);
    m_Req.u.data.length = sizeof (Range);
    m_Req.u.data.flags = 0;
    Ioctl (SIOCGIWRANGE, "Can't get supported rates.");
    for (unsigned i (0) ; i < Range.num_bitrates ; ++i)
        RatesVec.push_back (Range.bitrate [i] / TxUnit);
    std::sort (RatesVec.rbegin(), RatesVec.rend());
#ifndef NDEBUG
    std::cerr << "Exit CWE17Driver::GetSupportedRates()\n";
#endif // NDEBUG

} // GetSupportedRates()
