/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "NetworkRequestThread.h"

#include <QNetworkRequest>
#include <QNetworkAccessManager>
#include <QtNetwork/QNetworkReply>
#include <QMutexLocker>
#include <QDebug>

NetworkRequestThread::NetworkRequestThread(const QUrl &url, UnityCancellable *cancellable, const HttpHeadersList& headers)
    : QThread(),
      m_url(url),
      m_cancellable(nullptr),
      m_cancelSigId(0),
      m_headers(headers),
      m_reply(nullptr),
      m_manager(nullptr)
{
    if (cancellable != nullptr) {
        m_cancellable = unity_cancellable_get_gcancellable(cancellable);
    }
}

NetworkRequestThread::~NetworkRequestThread()
{
    if (m_manager)
        m_manager->deleteLater();

    if (m_reply)
        m_reply->deleteLater();

    if (m_cancellable && m_cancelSigId > 0)
        g_cancellable_disconnect(m_cancellable, m_cancelSigId);
}

void NetworkRequestThread::run()
{
    qDebug() << "Remote request:" << m_url;
    QMutexLocker lock(&m_mutex); // m_reply has to be locked till processing ends
 
    if (m_cancellable) {
        m_cancelSigId = g_cancellable_connect(m_cancellable, G_CALLBACK(onCancelled), this, nullptr);
        if (m_cancelSigId == 0)
            return;
    }

    m_manager = new QNetworkAccessManager();   
    QNetworkRequest request(m_url);
    for (auto it = m_headers.begin(); it != m_headers.end(); it++) {
        request.setRawHeader((*it).first, (*it).second);
    }

    connect(m_manager, SIGNAL(finished(QNetworkReply *)), this, SLOT(queryDone(QNetworkReply *)));
    m_reply = m_manager->get(request);

    QThread::exec(); // enter event loop
}

QNetworkReply* NetworkRequestThread::getReply() const
{
    QMutexLocker lock(&m_mutex);
    return m_reply;
}

void NetworkRequestThread::queryDone(QNetworkReply* /* reply */)
{
    quit();
}

void NetworkRequestThread::cancel()
{
    if (m_reply) {
        qDebug() << "Search request canceled:" << m_url;
        m_reply->abort();
        quit();
    }
}

void NetworkRequestThread::onCancelled(GCancellable * /* cancellable */, NetworkRequestThread *networkThread)
{
    networkThread->cancel();
}
