/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "Utils.h"
#include <glib.h>
#include <deelistmodel.h>
#include <uuid.h>

GIcon* gfileIconFromUri(const QString &uri)
{
    auto iconFile = g_file_new_for_uri(uri.toUtf8().data());
    auto icon = g_file_icon_new(iconFile);
    g_object_unref(iconFile);
    return icon;
}

GVariant* hashtableToAsv(GHashTable *hashtable)
{
    GHashTableIter iter;
    gpointer key, value;

    std::vector<GVariant*> children(g_hash_table_size(hashtable));
    
    int n = 0;
    g_hash_table_iter_init(&iter, hashtable);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        children[n++] = g_variant_new_dict_entry(g_variant_new_string((gchar*)key), g_variant_new_variant((GVariant*)value));
    }

    return g_variant_new_array(G_VARIANT_TYPE("{sv}"), children.data(), children.size());
}

QVariantMap ghashtableToQVariantHash(GHashTable *hashtable)
{
    QVariantMap vmap;
    GHashTableIter iter;
    gpointer key, value;
    
    g_hash_table_iter_init(&iter, hashtable);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        vmap.insert((gchar*)key, DeeListModel::VariantForData((GVariant*)value));
    }

    return vmap;
}

/**
 Generate Time-UUID with node (MAC) bytes replaced with random bytes, so that it doesn't contain any sensitive data.
*/
QString randomizedTimeUuid()
{
    char res[37];
    
    unsigned char data1[16];
    uuid_generate_time (data1);
    
    unsigned char data2[16];
    uuid_generate_random (data2);

    // last 6 bytes are MAC, overwrite with random bytes and set multicast bit, see section 4.5 of RFC 4122.
    for (int i = 10; i<16; i++)
        data1[i] = data2[i];
    data1[10] |= 1;
    
    uuid_unparse (data1, res);
    return QString(res);
}

QVariantHash QVariantMapToHash(const QVariantMap &map)
{
    auto result = QVariantHash();

    for (auto it = map.constBegin(); it != map.constEnd(); ++it) {
        if ((*it).type() == QVariant::Map) {
            auto val = QVariantMapToHash((*it).toMap());
            result.insert(it.key(), val);
        } else {
            result.insert(it.key(), it.value());
        }
    }

    return result;
}
